/*
	STB Systems Clock Chip Program

	Copyright 1983 by:
		STB Systems, Inc.
		1701 N Greenville,  Suite 703
		Richardson, TX 75081

	Rev 1.1 - 1 Aug 83 - Added monochrome card clock addr

*/

#include "stdio.h"

#define inp(x) inportb(x)
#define out(x,y) outportb(x,y)

int masks[] = {1, 2, 4, 8, 16, 32, 64, 128, 256, 512, 1024}; /* Bit masks */
int clkadrs[] = {0x37b, 0x27b, 0x3bf, 0};  /* Chip I/O addresses */
int wa[] = {0, 12, 8, 4, 0, 0, 6};	   /* Chip time-write codes */
int ra[] = {0, 12, 8, 4, 0, 6, 0};	   /* Chip time-read codes */

int clb;	/* Clock bit */
int dlen;	/* Data enable bit */
int clk;	/* Clock chip I/O address */

int hours, minutes, seconds, month, day, year;

main(argc,argv)
int argc;
char *argv[];
{
   int i;
   char c;
   int dt[7];


   findchip();

   if(argc <= 1) {	/* No params on command line */
      readchip();
      if (setdt(hours,minutes,seconds,month,day,year)) {
	 cls();
	 print("\n Time chip contains invalid data.");
	 print("\n Check the clock battery and its connections.\n");
	 askdt();
	 setchip();
      }
   }
   else {	/* Get date and time from command line */
      hours = minutes = seconds = 0;
      month = 1; day = 1; year = 1980;
      for(i=1; i <= (argc-1); i++) {
	 c = getcmdl(argv[i], dt);
	 if(c == ':') { hours = dt[0]; minutes = dt[1]; seconds = dt[2];}
	 if(c == '/') { month = dt[0]; day = dt[1]; year = dt[2];}
      }
      if(year < 100) year += 1900;
      setchip();
   }

   cls();
   print(" The date is ");
   putnum(month,'/'); putnum(day,'/'); putnum(19,' '); putnum(year-1900,' ');
   print(" at ");
   putnum(hours,':'); putnum(minutes,':'), putnum(seconds,'.');
   print("\n\n");
}

findchip()	/* Find the address the time chip is at */
{
   int i;

   clb = 2;	/* Clock chip clock bit mask */
   dlen = 4;	/* Clock chip data enable mask */

   i=0;
   while(clk = clkadrs[i++]) {
      resync();
      out(clk, dlen+1); 		/* Send a 1 */
      if ((inp(clk) & 1) == 1) continue;	/* Expect a 0 */
      out(clk, dlen+0); 		/* Send a 0 */
      if ((inp(clk) & 1) != 0) break;	/* Expect a 1 */
   }
   if (clk == 0) clk = clkadrs[0];	/* Default address */
}

resync()	/* Put time chip into a known state */
{
   int i;

   for (i=0; i<16; i++) {
      out(clk,0);
      out(clk,clb);
      out(clk,0);
   }
}

setchip()   /* Ask for date & time and set the time chip AND system clock */
{
   int x;	/* Temp value */
   int bit;	/* Bit counter */
   int i;	/* Temp counter */
   int dt[7];	/* Scratch array */

   x = minutes/10;		/* Convert binary to BCD */
   dt[1] = (x << 4) + minutes - x*10;
   x = hours/10;
   dt[2] = (x << 4) + hours - x*10;
   x = day/10;
   dt[3] = (x << 4) + day - x*10;
   x = month/10;
   dt[4] = (x << 4) + month - x*10 + 64;
   dt[5] = 0;
   x = (year - 1980) / 10;
   dt[6] = (x << 4) + year - 1980 - x*10;

   resync();

   for (i=1; i<=6; i++) {
      dt[i] = (dt[i] << 4) + wa[i];	/* Merge address and data */
      for(bit=0; bit<=10; bit++) {	/* Send adr-data bits */
	 x = (dt[i] & masks[bit]) == 0;
	 if (x) x = 1;
	 out(clk, dlen+x);
	 out(clk, dlen+x+clb);
	 out(clk, dlen+x);
      }
      out(clk, 0);	/* Strobe in the time */
      out(clk, clb);
      out(clk,0);
   }
}


askdt()
{
   do {
      print("\n Enter the date (MM/DD/YY): ");
      month = getnum();
      day = getnum();
      year = getnum();
      if(year < 100) year += 1900;

      print("\n Enter the time (HH:MM): ");
      hours = getnum();
      minutes = getnum();
      seconds = 0;
   }  while (setdt(hours, minutes, seconds, month, day, year));

}

getcmdl(s, d)	/* Decode command line argument "s" into array "d" */
char s[];
int d[];
{
   char c, term;
   int i, dp, val;

   d[0] = d[1] = d[2] = i = dp = val = 0;
   while(1) {
      c = s[i++];
      if (c == EOS) break;
      if ((c == ':') || (c == '/')) {
	 if(dp > 2) continue;
	 term = c;
	 d[dp++] = val; val = 0;
	 continue;
      }
      if ((c >= '0') && (c <= '9')) val = val * 10 + c - '0';
      else {
	 print("\n\n Invalid command usage. ");
	 print("\n To set the clock chip: STBCLOCK MM/DD/YY HH:MM");
	 print("\n To set the system clock: STBCLOCK");
	 print("\n\n");
	 exit(-1);
      }
   }
   d[dp++] = val;
   return term;
}



readchip()	/* Read the time chip into the date & time variables */
{
   int i;
   int x;
   int bit;
   int dt[7];	/* Scratch array */


   resync();

   for (i=1; i<=5; i++) {
      for (bit=0; bit<=3; bit++) {	/* Send address to chip */
	 x = (ra[i] & masks[bit]) == 0;
	 if (x) x = 1;
	 out(clk, dlen+x);		/* Strobe in the address bit */
	 out(clk, dlen+x+clb);
	 out(clk, dlen+x);
      }

      out(clk,0);	/* Strobe in the address */
      out(clk,clb);
      out(clk,0);

      dt[i] = 0;

      for(bit=0; bit<=5; bit++) {	/* Read the time regs */
	 out(clk, dlen);
	 dt[i] += ((inp(clk) & 1) << bit);
	 out(clk, dlen+clb);
	 out(clk, dlen);
      }

      dt[i] += ((inp(clk) & 1) << 6);
      out(clk, 0);	/* Done with read cycle */
      dt[i] = (dt[i] >> 4)*10 + (dt[i] & 0x0F);  /* BCD to decimal */

   }
   dt[5] += 1980;

   hours = dt[2];
   minutes = dt[1];
   seconds = 0;
   month = dt[4];
   day = dt[3];
   year = dt[5];

}

setdt (hours, minutes, seconds, month, day, year)
int    hours, minutes, seconds, month, day, year;
{
   /* Set system clock - MS DOS Version */
   /* Error code return: 0 - OK, 1 - Bad time, 2 - Bad date, 3 - Both Bad */

   int err;
   struct segs { int scs, sss, sds, ses;} sreg; /* Segment regs */
   struct regs { int ax, bx, cx, dx, si, di, ds, es;} reg;


   segread(&sreg);			/* Read the segment regs */
   reg.ds = sreg.sds;
   reg.es = sreg.ses;

   reg.cx = (hours << 8) + minutes;	/* HH:MM */
   reg.dx = seconds;			/* seconds */
   reg.ax = 0x2d << 8;
   sysint(0x21, &reg, &reg);		/* Set the time */

   err = 0;
   if(reg.ax & 0xFF) err=1;		/* If bad time from chip */

   reg.cx = year;			/* Year */
   reg.dx = (month << 8) + day; 	/* MM/DD */
   reg.ax = 0x2b << 8;
   sysint(0x21, &reg, &reg);		/* Set the date */
   if(reg.ax & 0xff) err += 2;

   return err;
}

print(s)
char s[];
{
int i;
char c;

   i=0;
   while(c=s[i++]) {
      co(c);
      if(c == '\n') co(0x0D);
   }
}

putnum(num,term)	/* Print a two digit decimal number */
int num;
char term;
{
   if ((num > 99) || (num < 0)) {
      print("??");
   }
   else {
      co((num / 10) + '0');
      co((num % 10) + '0');
   }
   if(term != ' ') co(term);
}

getnum()	/* Read a decimal number from the keyboard */
{
   int val;
   char c;

   val=0;
   while(1) {
      c=ci(); co(c); if(c == 0x0D) co('\n');
      if ((c < '0') || (c > '9')) break;        /* Any non-numeric exits */
      val = val * 10 + c - '0';
   }
   return val;
}

cls()		/* Home and clear the screen */
{
   struct segs { int scs, sss, sds, ses;} sreg; /* Segment regs */
   struct regs { int ax, bx, cx, dx, si, di, ds, es;} reg;

   segread(&sreg);		/* Read the segment regs */
   reg.ds = sreg.sds;
   reg.es = sreg.ses;

   reg.ax = (15 << 8);
   sysint (0x10, &reg, &reg);	/* Get video mode */

   reg.ax &= 0xff;
   sysint (0x10, &reg, &reg);	/* Reset mode to clear screen & home */

}
